
#include "yuv_video.h"


/*
 * yuv_video::yuv_video(const char *file, const int x, const int y, const bool write)
 * Constructor, when file name, picture format, and mode of operation are known
 * This is a public method
 *
 * Requires:	Valid file name, frame size in X and Y axis, and mode of operation
 * Modifies:	_horiz, _vert, _filename, _initialized, _y, _u, _v, _writing
 * Ensures:		._horiz set to x, _horize set to y
 *				.before calling yuv_video:Init(void)
 *					- set _y, _u, _v, _filename to NULL
 *					- set _writig to -1
 *					- set _initialized to false 
 *				.call yuv_video:Init(void) 
 *
 * Alert:		Should cause program to terminate if given invalid inputs
 */
yuv_video::yuv_video(const char *filename, const int x, const int y, const bool write)
{
  _horiz = x;
  _vert = y;
  _y = NULL;
  _u = NULL;
  _v = NULL;
  _writing = (int) write;
  _filename = (char *) strdup(filename);
  _fp = NULL;
  _initialized = false;
  _first_write = false;

  Init();
}


/*
 * yuv_video::yuv_video(const int x, const int y)
 * Constructor, when only picture format is known
 * This is a public method
 *
 * Requires:	Valid frame size in X and Y axis
 * Modifies:	_horiz, _vert, _filename, _initialized, _y, _u, _v, _writing
 * Ensures:		._horiz set to x, _horize set to y
 *				.before calling yuv_video:Init(void)
 *					- set _y, _u, _v, _filename to NULL
 *					- set _writig to -1
 *					- set _initialized to false 
 *				.call yuv_video:Init(void) 
 *
 * Alert:		Should cause program to terminate if given invalid inputs
 */
yuv_video::yuv_video(const int x, const int y)
{
  _horiz = x;
  _vert = y;

  _filename = NULL;
  _fp = NULL;
  _initialized = false;
  _y = NULL;
  _u = NULL;
  _v = NULL;
  _writing = -1;
  _first_write = false;

  Init();
}


/*
 * yuv_video::yuv_video(void)
 * Constructor, when nothinhg is known
 * This is a public method
 *
 * Requires:	nothing
 * Modifies:	_horiz, _vert, _filename, _initialized, _y, _u, _v, _writing
 * Ensures:		._horiz set to -1, _horize set to -1
 *				._filename set to NULL
 *				._y, _u, _v set to NULL
 *				._writing set to -1
 *				.call yuv_video:Init(void) 
 *
 */
yuv_video::yuv_video(void)
{

	_horiz = -1;
	_vert = -1;
	_filename = NULL;
	_fp = NULL;
	_initialized = false;
	_y = NULL;
	_u = NULL;
	_v = NULL;
	_writing = -1;
	_first_write = false;
	
	Init();
}


/*
 * yuv_video::yuv_video(void)
 * Constructor, when nothinhg is known
 * This is a public method
 *
 * Requires:	nothing
 * Modifies:	_horiz, _vert, _filename, _initialized, _y, _u, _v, _writing
 * Ensures:		._horiz set to -1, _horize set to -1
 *				._filename set to NULL
 *				._y, _u, _v set to NULL
 *				._writing set to -1
 *				.call yuv_video:Init(void) 
 *
 */
yuv_video::yuv_video(const yuv_video &y)
{
	// Have to copy everything!
	_horiz = y._horiz;
	_vert = y._vert;
	
	if(y._filename != NULL)
		_filename = (char *) strdup(y._filename);
	else
		_filename = NULL;
	
	_initialized = false;
	_writing = y._writing;
	_first_write = y._first_write;
	
	// Hopefully this will re-read the data.
	Init();
}


/*
 * yuv_video::~yuv_video()
 * Destructor for yuv_video object
 * This is a public method
 *
 * Requires:	A valid object, with _fp opened
 * Modifies:	_fp, _filename, _y, _u, _v, *this
 * Ensures:		. _fp closed
 *				. delete[] _filename, _y, _u, _v
 *				. the object is destroyed
 *
 * Alert:	Check on _writing part of code
 */
yuv_video::~yuv_video()
{
	if(_writing == 1 && _fp != NULL) WriteVideo();	// Have we finished the frame?
	if (_fp != NULL) fclose(_fp);

	delete[] _filename;
	delete[] _y;
	delete[] _u;
	delete[] _v;
}


/*
 * void yuv_video::Init(void)
 * Initialization of data members
 * This is a private method
 * 
 *
 * Requires:	valid _filename
 * Modifies:	_y, _u, _v, _tu, _tv, cur_frame, , _size, _initialized, _fp
 * Ensures:		.DO NOTHING if _horiz, _vert, or _writing is -1
 *				.memory is allocated for _y, _u, _v
 *				. _cur_frame set to 0, _size set to frame size (Y+U+V)
 *				._fp opened in "wb" if _writing, else in "rb"; exit(-1) if cannot open file
 *				.call yuv_video:ReadVideo() if in reading mode
 *				.if no error from ReadVideo, _initialized set to true
 * 
 */
void yuv_video::Init(void)
{
	// Check all data.
	if(_horiz <= 0) return;
	if(_vert <= 0) return;
	if(_writing == -1) return;	// _filename available if _writing is true
	
	// Satisfied with basic data, move on with the rest of task
	_y = new unsigned char[_horiz * _vert];
	_u = new unsigned char[(_horiz / 2) * (_vert / 2)];
	_v = new unsigned char[(_horiz / 2) * (_vert / 2)];
	
	_cur_frame = 0;
	
	// sizeof (single frame) = sizeof (Y frame) + size of (U frame) + size of (V frame)
	_size = (_horiz * _vert) + ((_horiz / 2) * (_vert / 2) * 2);
	
	if(_writing == 1) {
		// Writing
		if(_filename == NULL) exit(-1);
		else {
			if((_fp = fopen(_filename, "wb")) == NULL) {
				cerr << "Can't open file " << _filename << " for writing!\n";
				exit(-1);
			}
		}

	} else {
		// Reading
		if(_filename == NULL) exit(-1);
		else {
			if((_fp = fopen(_filename, "rb")) == NULL) {
				cerr << "Can't open file " << _filename << " for reading!" << endl;
				exit(-1);
			}
		}
		// Read the initial frame.
		ReadVideo();
	}
	_initialized = true;
}


/*
 * void yuv_video::ReadVideo(void)
 * Read a video frame from a binary file
 * Thus is a private method
 * 
 * Requires:	_fp is ready for reading in binary format
 * Modifies:	_y, _u, _v
 * Ensures:		Y frame is read into _y
 *				U frame is read into _u
 *				V frame is read into _v
 *				
 * Alert:		Should throw an exception if an error occurs here
 */
void yuv_video::ReadVideo(void)
{

	if (_fp == NULL) {
		cerr << "File pointer is null, unable to read";
		exit(-1);
	}
	
	// Read "Y" component, size _horiz*_vert
	fread(_y, sizeof(unsigned char), _vert * _horiz, _fp);
	
	// Read "U" component, size _horiz*_vert/4
	fread(_u, sizeof(unsigned char), (_vert*_horiz>>2), _fp);
	
	// Read "V" component, size _horiz*_vert/4
	fread(_v, sizeof(unsigned char), (_vert*_horiz>>2), _fp);
}



/*
 * void yuv_video::WriteVideo(void)
 * Write a video frame into a binary file
 * Thus is a private method
 * 
 * Requires:	_fp is ready for writing in binary format
 * Modifies:	nothing (except _fp)
 * Ensures:		_y is written to _fp
 *				_u is written to _fp
 *				_v is written to _fp
 *				
 * Alert:		Should throw an exception if an error occurs here
 */
void yuv_video::WriteVideo(void)
{
	if (_first_write) {
		// First, the "y" component
		fwrite(_y, sizeof(unsigned char), _vert * _horiz, _fp);
		// Next, the "u" component
		fwrite(_u, sizeof(unsigned char), (_vert / 2) * (_horiz / 2), _fp);
		// And finally, the "v" component
		fwrite(_v, sizeof(unsigned char), (_vert / 2) * (_horiz / 2), _fp);
	}
}


/*
 * void yuv_video::FetchFrame(const int frame_index)
 * Fetch a frame from the file stream
 * This is a private method
 *
 * Requires:	_fp is ready for reading in binary mode
 *				file pointer is at the beginning of next frame, instead of _cur_frame
 *				frame_index indicating the desired frame in pictures
 * Modifies:	_y, _u, _v, _cur_frame, _fp (location of the file pointer)
 * Ensures:		_y, _u, _v, respectively set to the Y, U, V "frame" of the desired frame in the picture
 *				_fp is set to the beginning of the (frame_index+1)-th frame
 *				_cur_frame set to _frame_index
 *				
 * Alert:		Unroll the necessary loops and reduce the method call frequencies
 */
void yuv_video::FetchFrame(const int frame_index)
{
	if(_cur_frame < frame_index) {
		while(_cur_frame != frame_index) {
			ReadVideo();
			_cur_frame++;
		}
	} else {
		// FP is at the head of (_cur_frame+1)-th frame, 
		// as its contents are stored in this object
		while(_cur_frame != frame_index ) {
			fseek(_fp, -2*_size, SEEK_CUR);
			_cur_frame--;
			ReadVideo();	// This will move fp ahead by 1 frame
		}
	}
}


/*
 * void yuv_video::DumpFrame(const int frame_index)
 * The purpose is to adjust the file pointer to (n+1)th frame as
 *		we move the _cur_frame to the nth frame
 * This is a private method
 *
 * Requires:	_fp is ready for writing in binary mode
 *				file pointer is at the beginning of next frame, instead of _cur_frame
 *				frame_index indicating the desired frame in pictures
 * Modifies:	_cur_frame, _fp (location of the file pointer)
 * Ensures:		.if _fp is NULL, exit(-1)
 *				.if _cur_frame is too small, call WriteVideo() then _cur_frame++
 *				.otherwise, call fseek then _cur_frame--, continue until done
 *
 * Note:		_fp is set to the beginning of the (frame_index+1)th frame*				
 * Alert:		This is probably never called
 */
void yuv_video::DumpFrame(const int frame_index)
{
	if (_fp == NULL) {
		cerr << "Dumping Frame, but file pointer is emply\n";
		exit(-1);
	}

	if(_cur_frame < frame_index) {
		while(_cur_frame != frame_index) {	// this loop does not make sense
			WriteVideo();
			_cur_frame++;
		}
	} else {
		WriteVideo();	// Dump to the file. The dumped data is ignored, overwritten.....
		fseek(_fp, -_size, SEEK_CUR);
		while(_cur_frame != frame_index) {	// only after _fp?
			fseek(_fp, -_size, SEEK_CUR);
			_cur_frame--;
		}
	}
}


/*
 * unsigned char *yuv_video::Get_Y(const int frame_index)
 * Return the "Y" component.
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		if _cur_frame != frame_index, call FetchFrame(frame_index)
 *				return _y
 *
 */
unsigned char *yuv_video::Get_Y(const int frame_index)
{
	if(!_initialized) {
		cerr << "Can't get frame from unitialized video!\n";
		exit(-1);
	}

	if (_cur_frame != frame_index) FetchFrame(frame_index);

	return _y;
}


/*
 * unsigned char *yuv_video::Get_Y(const int frame_index, const int x, const int y)
 * Return the "Y" component, at (x, y) position
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		call Get_Y(frame_index)
 *				if x or y invalid, cerr and terminate
 *				return _y[x + y*_horiz]
 *
 * Warning:		This method is currently not used
 */
unsigned char yuv_video::Get_Y(const int frame_index, const int x, const int y)
{
	Get_Y(frame_index);
	
	if(x > _horiz || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << _horiz << endl;
		exit(-1);
	}
	
	if(y > _vert || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and " << _vert << endl;
		cerr << "Can't read y = " << x << endl;
		exit(-1);
	}
	
	return _y[x + y * _horiz];
}


/*
 * unsigned char *yuv_video::Get_U(const int frame_index)
 * Return the "U" component.
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		if _cur_frame != frame_index, call FetchFrame(frame_index)
 *				return _u
 *
 */
unsigned char *yuv_video::Get_U(const int frame_index)
{
	if(!_initialized) {
		cerr << "Can't get frame from unitialized video!\n";
		exit(-1);
	}

	if (_cur_frame != frame_index) FetchFrame(frame_index);
	
	return _u;
}


/*
 * unsigned char *yuv_video::Get_U(const int frame_index, const int x, const int y)
 * Return the "U" component, at (x, y) position
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		call Get_U(frame_index)
 *				if x or y invalid, cerr and terminate
 *				*note* bounds for x and y is 1/2 of _horiz and _vert respectively
 *				return _u[x + y*_horiz/2]
 *
 * Warning:		This method is currently not used
 */
unsigned char yuv_video::Get_U(const int frame_index, const int x, const int y)
{
	Get_U(frame_index);
	
	if(x > (_horiz>>1) || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << (_horiz>>1) << endl;
		exit(-1);
	}
	
	if(y > (_vert>>1) || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and  "<< (_vert>>1) << endl;
		cerr << "Can't read y = " << x << endl;
		exit(-1);
	}
	
	return _u[x + ((y * _horiz) >> 1)];
}


/*
 * unsigned char *yuv_video::Get_V(const int frame_index)
 * Return the "V" component.
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		if _cur_frame != frame_index, call FetchFrame(frame_index)
 *				return _v
 *
 */
unsigned char *yuv_video::Get_V(const int frame_index)
{
	if(!_initialized) {
		cerr << "Can't get frame from unitialized video!\n";
		exit(-1);
	}

	if (_cur_frame != frame_index) FetchFrame(frame_index);
	
	return _v;
}



/*
 * unsigned char *yuv_video::Get_V(const int frame_index, const int x, const int y)
 * Return the "V" component, at (x, y) position
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	nothing
 * Ensures:		call Get_V(frame_index)
 *				if x or y invalid, cerr and terminate
 *				*note* bounds for x and y is 1/2 of _horiz and _vert respectively
 *				return _v[x + y*_horiz]
 *
 * Warning:		This method is currently not used
 */
unsigned char yuv_video::Get_V(const int frame_index, const int x, const int y)
{
	Get_V(frame_index);
	
	if(x > (_horiz>>1) || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << (_horiz>>1) << endl;
		exit(-1);
	}
	
	if(y > (_vert>>1) || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and " << (_vert>>1) << endl;
		exit(-1);
	}
	
	return _v[x + ((y * _horiz) >> 1)];
}


/*
 * void yuv_video::Put_Y(const int frame_index, const int x, const int y, const unsigned char val)
 * Write a value to (x,y) in _y in this object
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 *				valid x, y
 * Modifies:	_y
 * Ensures:		_y[x+y*_horiz] = val
 *
 */
void yuv_video::Put_Y(const int frame_index, const int x, const int y, const unsigned char val)
{
	if (_cur_frame != frame_index) DumpFrame(frame_index);
	
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!\n";
		exit(-1);
	}
	
	if(x > _horiz || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << _horiz << endl;
		exit(-1);
	}
	
	if(y > _vert || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and " << _vert << endl;
		exit(-1);
	}

	_y[x + y * _horiz] = val;
}


/*
 * void yuv_video::Put_U(const int frame_index, const int x, const int y, const unsigned char val)
 * Write a value to (x,y) in _u in this object
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 *				valid x, y
 * Modifies:	_u
 * Ensures:		_u[x+y*_horiz/2] = val
 *
 */
void yuv_video::Put_U(const int frame_index, const int x, const int y, const unsigned char val)
{
	if (_cur_frame != frame_index) DumpFrame(frame_index);
	
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!" << endl;
		exit(-1);
	}
	
	if(x > (_horiz>>1) || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << (_horiz>>1) << endl;
		exit(-1);
	}
	
	if(y > (_vert>>1) || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and " << (_vert>>1) << endl;
		exit(-1);
	}

    
	_u[x + (y * _horiz) >> 1] = val;
}


/*
 * void yuv_video::Put_V(const int frame_index, const int x, const int y, const unsigned char val)
 * Write a value to (x,y) in _v in this object
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 *				valid x, y
 * Modifies:	_v
 * Ensures:		_v[x+y*_horiz/2] = val
 *
 */
void yuv_video::Put_V(const int frame_index, const int x, const int y, const unsigned char val)
{
	if (_cur_frame != frame_index) DumpFrame(frame_index);
	
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!" << endl;
		exit(-1);
	}
	
	if(x > (_horiz>>1) || x < 0) {
		cerr << "Out of bound: x=" << x << "is not between 0 and " << (_horiz>>1) << endl;
		exit(-1);
	}
	
	if(y > (_vert>>1) || y < 0) {
		cerr << "Out of bound: y=" << y << "is not between 0 and " << (_vert>>1) << endl;
		exit(-1);
	}
    
	_v[x + (y * _horiz) >> 1] = val;
}


/*
 * void yuv_video::Set_Y(const int frame_index, const unsigned char *y)
 * copy contents of y to _y
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	_y
 * Ensures:		memcpy contents y to _y
 *
 */
void yuv_video::Set_Y(const int frame_index, const unsigned char *y)
{
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!" << endl;
		exit(-1);
	}

	_first_write = true;
	if (_cur_frame != frame_index) DumpFrame(frame_index);
	memcpy(_y, y, sizeof(unsigned char) * _horiz * _vert);
}  


/*
 * void yuv_video::Set_V(const int frame_index, const unsigned char *u)
 * copy contents of u to _u
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	_u
 * Ensures:		memcpy contents of u to _u
 *
 */
void yuv_video::Set_U(const int frame_index, const unsigned char *u)
{
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!" << endl;
		exit(-1);
	}

	_first_write = true;
	if (_cur_frame != frame_index) DumpFrame(frame_index);	
	memcpy(_u, u, (sizeof(unsigned char) * _horiz * _vert) >> 2 );
}  


/*
 * void yuv_video::Set_V(const int frame_index, const unsigned char *v)
 * copy contents of v to _v
 * This is a public method
 *
 * Requires:	frame_index not out of bound, ie. smalller than 0 or bigger than the picture's total frames
 * Modifies:	_v
 * Ensures:		memcpy contents of v to _v
 *
 */
void yuv_video::Set_V(const int frame_index, const unsigned char *v)
{
	if(!_initialized) {
		cerr << "Can't put frame to unitialized video!" << endl;
		exit(-1);
	}

	_first_write = true;
	if (_cur_frame != frame_index) DumpFrame(frame_index);
	memcpy(_v, v, sizeof(unsigned char) * _horiz / 2 * _vert / 2);
}  


/*
 * void yuv_video::SetFile(const char *filename, const bool writing)
 * Set the filename as well as mode of operation
 * This is a public method
 *
 * Requires:	filename is the name of the file and not NULL
 *				writing is mode of operation, reading or writing
 * Modifies:	_filename, _writing
 * Ensures:		.if _filename is NULL, exit(-1)
 *				._filename set to filename
 *				._writing set to writing
 *				.call Init()
 *
 */
void yuv_video::SetFile(const char *filename, const bool writing)
{
	if(filename == NULL) exit(-1);	// illegal usage
	else _filename = strdup(filename);
	_writing = (int) writing;

	Init();
}


/*
 * void yuv_video::SetXY(const int x, const int y)
 * Set dimension of a frame
 * This is a public method
 *
 * Requires:	x and y dimension of a frame
 * Modifies:	_horitz, _vert
 * Ensures:		._horiz set to x, _vert set to y
 *				.call Init()
 *
 * Alert:		this method is not being used
 */
void yuv_video::SetXY(const int x, const int y)
{
	_horiz = x;
	_vert = y;
	
	Init();
}
