#ifndef __TREE_C__
#define __TREE_C__

#include "tree.h"

/*
 * Tree::Tree(const short int value, const char* ptr_c)
 *
 * Constructor to add children onto a node give a string
 *
 * Requires:
 * value holds the index of the entry ptr_c
 *
 * Modifies:
 * Depending on what *ptr_c is, that child branch will recursively
 * construct the remaining bits described in ptr_c
 * _isvalid notifies if the node is a leaf to save on NULL checking
 *
 * Ensures:
 * Fast and easy lookup of indicies from an incoming bitstream
 */
Tree::Tree(const short int value, const char* ptr_c) {
	child[0] = NULL;
	child[1] = NULL;
	if (strlen(ptr_c) > 0) {
		_isvalid = false;
		if (*ptr_c++ == '1') child[1] = new Tree(value, ptr_c);
		else child[0] = new Tree(value, ptr_c);
	}
	else {
		_isvalid = true;
		entry = value;
	}
}

/*
 * Tree::Tree(const char** const table, const int num_values)
 *
 * Creates binary search tree from the root
 *
 * Requires:
 * table which is at least num_values large holding strings containing
 * bitstreams. The index of each string is gathered from the index of
 * the string in the table
 *
 * Modifies:
 * This will loop through all the entries of the table.
 * Depending on what *ptr_c is, that child branch will recursively
 * construct the remaining bits described in ptr_c
 * _isvalid notifies if the node is a leaf to save on NULL checking
 *
 * Ensures:
 * Fast and easy lookup of indicies from an incoming bitstream
 */
Tree::Tree(const char** const table, const int num_values) {
	short int i;
	const char* ptr_c;

	_isvalid = false;
	child[0] = NULL;
	child[1] = NULL;
	for (i = 0; i < num_values; i++) {
		ptr_c = table[i];
		Insert_Entry(i, ptr_c);  // Entries start from 0
	}
}

/*
 * Tree::~Tree()
 *
 * Destructor
 *
 * Requires: nothing
 *
 * Modifies:
 * deletes nodes recursively if nodes are not NULL
 *
 * Ensures:
 * Proper deletion of dynamically allocated variables
 */
Tree::~Tree() {
	if (child[0] != NULL) delete child[0];
	if (child[1] != NULL) delete child[1];

}

/*
 * Tree::Insert_Entry(const short int value, const char* ptr_c)
 *
 * Adds children given the string
 *
 * Requires:
 * value which is the entry for the remaining sequence ptr_c
 * _isvalid to be false for the resulting leaf of sequence ptr_c
 *
 * Modifies:
 * If child does not exist, it will recursively call the constructor to create one
 * If the child exits, it will recursively insert the value for the
 * remaining sequence ptr_c
 * If this is the leaf, deposit value as the entry
 *
 * Ensures:
 * Fast and easy lookup of indicies from an incoming bitstream
 */
void Tree::Insert_Entry(const short int value, const char* ptr_c) {
	if (strlen(ptr_c) > 0) {
		if (*ptr_c++ == '1') {
			if (child[1] == NULL) child[1] = new Tree(value, ptr_c);
			else child[1]->Insert_Entry(value,ptr_c);
		}
		else {
			if (child[0] == NULL) child[0] = new Tree(value, ptr_c);
			else child[0]->Insert_Entry(value,ptr_c);
		}
	}
	else {
		if (_isvalid) cout << "table not unique!" << endl;
		_isvalid = true;
		entry = value;
	}
}

/*
 * Tree::Read_Entry(short int* value, PSC_Fifo &input)
 *
 * Traverse tree to obtain the entry
 *
 * Requires:
 * value is where the result will be deposited into
 * input must be a valid input bitstream
 *
 * Modifies:
 * If the node is a leaf, *value = entry
 *
 * Ensures:
 * Fast and easy lookup of indicies from an incoming bitstream
 */
bool Tree::Read_Entry(short int* value, PSC_Fifo &input) {
	if (_isvalid) {
		*value = entry;
		return (true);
	}
	else {
		bool bit = input.ReadBit();
		if (child[bit] != NULL) {
			return (child[bit]->Read_Entry(value, input));
		}
	}
	return (false);
}

#endif