
#ifndef __PSC_FIFO_C__
#define __PSC_FIFO_C__

#include "PSC_Fifo.h"


/*
 * PSC_Fifo::PSC_Fifo(ifstream *fileptr)
 *
 * Constructor for reading an input file bitstream
 *
 * Requires : 
 * A valid and open ifstream file handle
 *
 * Modifies :
 * _mode is set to false,
 * _bitstream holds a new Bitstream
 * _buffer holds a buffer for Forward Error Correction
 */
PSC_Fifo::PSC_Fifo(ifstream *fileptr) {
	_bitstream = new Bitstream(fileptr);
	_buffer = new bool[_buffer_size];
	_mode = false;
	_SAC_mode = false;
	_found_header = false;
	_zerorun = 0;
}


/*
 * PSC_Fifo::PSC_Fifo(ofstream *fileptr)
 *
 * Constructor for writing to an output file bitstream
 *
 * Requires : 
 * A valid and open ofstream file handle
 *
 * Modifies :
 * _mode is set to true,
 * _bitstream holds a new Bitstream
 * _buffer holds a buffer for Forward Error Correction
 */
PSC_Fifo::PSC_Fifo(ofstream *fileptr) {
	_bitstream = new Bitstream(fileptr);
	_buffer = new bool[_buffer_size];
	_mode = true;
	_found_header = false;
	_SAC_mode = false;
	_zerorun = 0;
}


/*
 * PSC_Fifo::~PSC_Fifo()
 *
 * Destructor of PSC_Fifo class
 *
 * Requires : 
 * nothing
 *
 * Modifies :
 * Deletes bitstream
 * Deletes dynamically allocated buffer
 *
 * Ensures :
 * Prevention of memory leaks
 */
PSC_Fifo::~PSC_Fifo() {
	delete _bitstream;
	delete [] _buffer;
}


/*
 * void PSC_Fifo::Enable_SAC_Zero_Run()
 *
 * Notifies PSC_Fifo that bits are SAC
 *
 * Requires : 
 * nothing
 *
 * Modifies :
 * Sets _SAC_mode
 */
void PSC_Fifo::Enable_SAC_Zero_Run() {
	if(!_SAC_mode) {
		_SAC_mode = true;
		_zerorun = 0;
	}
}


/*
 * void PSC_Fifo::Disable_SAC_Zero_Run()
 *
 * Notifies PSC_Fifo that bits are VLC
 *
 * Requires : 
 * nothing
 *
 * Modifies :
 * Resets _SAC_mode
 */
void PSC_Fifo::Disable_SAC_Zero_Run() {
	_SAC_mode = false;
}


/*
 * void PSC_Fifo::PeekNBits(bool *bits, int N)
 *
 * Peeks at N Bits from input stream
 *
 * Requires :
 * *bits holds the output of this function (must be at least N large)
 * N is how many bits to peeked
 *
 * Modifies :
 * *bits will hold the next N bits.
 */
void PSC_Fifo::PeekNBits(bool *bits, const int N) {
	_bitstream->PeekNBits(bits, N);
}


/*
 * void PSC_Fifo::ReadNBits(bool *bits, int N)
 *
 * Reads N Bits from input stream
 *
 * Requires :
 * *bits holds the output of this function (must be at least N large)
 * N is how many bits to read
 *
 * Modifies :
 * *bits will hold the next N bits.
 * 
 * Ensures :
 * Allows user to read a number of bits
 */
void PSC_Fifo::ReadNBits(bool *bits, const int N) {
	int i;
	
	if (_SAC_mode) {
		for (i = 0; i < N; i++) {
			if (_zerorun == 15) {
				for (;i < N; i++) *bits++ = false;
				return;
			}
			else if (_zerorun == 14) {
				if (!_bitstream->ReadBit()) {
					// Must be a header!
					_zerorun++;
					for (;i < N; i++) *bits++ = false;
					return;
				} else _zerorun = 0;
			}
			_bitstream->ReadNBits(bits, 1);
			if (!*bits++) _zerorun++;
			else _zerorun=0;
		}
	}
	else _bitstream->ReadNBits(bits, N);
}


/*
 * bool PSC_Fifo::ReadBit()
 *
 * Reads one bit from stream
 *
 * Requires :
 * nothing
 *
 * Modifies :
 * nothing
 *
 * Ensures :
 * Allows user to read only one bit
 */
bool PSC_Fifo::ReadBit() {
	bool b;

	if (_SAC_mode) {
		if (_found_header) return false;
		else if (_zerorun == 14) {
			if (!_bitstream->ReadBit()) {
				// Must be a header!
				_zerorun++;
				return false;
			}
			else _zerorun = 0;
		}
		_bitstream->ReadNBits(&b, 1);
		if (!b) _zerorun++;
		else {
			if (_zerorun > 15) _found_header = true;
			else _zerorun = 0;
		}
	}
	else _bitstream->ReadNBits(&b, 1);
	return b;
}


/*
 * void PSC_Fifo::WriteNBits(bool *bits, int N)
 *
 * Writes N Bits into stream
 *
 * Requires :
 * *bits holds the bits to be written (must be at least N large)
 * N is how many bits are to be written
 *
 * Modifies :
 * 
 * Ensures :
 * Allows user to write a number of bits
 */
void PSC_Fifo::WriteNBits(const bool *bits, const int N) {
	int i;
	const bool *bitsptr;
	bool one;

	if (_SAC_mode) {
		bitsptr = bits;
		one = true;
		for (i = 0; i < N; i++, bitsptr++) {
			if (_zerorun == 14) {
				_bitstream->WriteNBits(&one,1);
				_zerorun = 0;
			}
			if (!*bitsptr) _zerorun++;
			else _zerorun = 0;

			_bitstream->WriteNBits(bitsptr,1);
		}
	}
	else _bitstream->WriteNBits(bits, N);
}


bool PSC_Fifo::FoundHeader(void) {
	return (_found_header);
}

int PSC_Fifo::GetZeroRun(void) {
	return _zerorun;
}

void PSC_Fifo::ResetFifo(void) {
	_zerorun = 0;
	_found_header = false;
	_SAC_mode = false;
}
#endif

