#ifndef __BITSTREAM_CPP__
#define __BITSTREAM_CPP__

#include "Serialencoder.h"
#include "Bitstream.h"

extern bool Encoder_Abort;
extern HANDLE ghWriterHeap;
extern CRITICAL_SECTION gcsWriterHeap;
extern CRITICAL_SECTION gcsDataHeap;
extern HWND ghWndStatusDlg;

// From transfer.cpp
extern HANDLE hTransferAbortEvent;
extern HANDLE hDataHeap;

/*
 * Bitstream::Bitstream(ifstream *fileptr)
 *
 * Constructor for reading an input file bitstream
 *
 * Requires : 
 * A valid and open ifstream file handle
 *
 * Modifies :
 * _mode is set to false, an input file stream
 * _infile points to the file handle
 *
 * Ensures :
 * Calls Init() to initialize and construct bitstream class for reading
 */
Bitstream::Bitstream(ifstream *fileptr) {
	_mode = false;	// This is an input file stream
	_infile = fileptr;
	
	Init();
}

/*
 * Bitstream::Bitstream(ofstream *fileptr)
 *
 * Constructor for writing to an output file bitstream
 *
 * Requires : 
 * A valid and open ofstream file handle
 *
 * Modifies :
 * _mode is set to true, an output file stream
 * _outfile points to the file handle
 *
 * Ensures :
 * Calls Init() to initialize and construct bitstream class for writing
 */
Bitstream::Bitstream(ofstream *fileptr) {
	_mode = true;	// This is an output file stream
	_outfile = fileptr;

	Init();
}

/*
 * Bitstream::~Bitstream()
 *
 * Destructor of bitstream class
 *
 * Requires : 
 * nothing
 *
 * Modifies :
 * If bitstream is an output file, flush out the buffer, even incomplete bytes
 * Deletes dynamically allocated buffer
 *
 * Ensures :
 * Proper flushing for output of buffer and prevention of memory leaks
 */
Bitstream::~Bitstream() {
	if (_mode) Flush(true);	// If writing, flush out everything

	delete [] _buffer;
}

/*
 * Bitstream::Init()
 *
 * Initializer of variables for bitstream class for use
 *
 * Requires : 
 * Uninitialized variables within bitstream class
 *
 * Modifies :
 * _buffer is dynamically allocated to hold _buffer_size bits
 * _eof signals that no end of file has been encountered
 *
 * Ensures :
 * Proper flushing for output of buffer and prevention of memory leaks
 */
void Bitstream::Init() {
	_buffer = new bool[_buffer_size];	// Create the buffer

	_eof = false;
	_current = 0;	// Where read from or write into the buffer, input and output respectively
	_cache_end = 0;	// Where write into or read from the buffer for file access, input and output respectively
	_cached_bits = 0;	// How many valid bits are cached
}

/*
 * void Bitstream::PeekNBits(bool *bits, int N)
 *
 * Peeks at N Bits from input stream
 *
 * Requires :
 * *bits holds the output of this function (must be at least N large)
 * N is how many bits to read (must be less than _buffer_size)
 *
 * Modifies :
 * *bits will hold the next N bits.
 * NOTE: this is a peek, so the first bit peeked will be the next bit read
 * Once EOF is reached, zeros will be padded into *bits
 * 
 * Ensures :
 * Allows user to peek a number of bits without advancing pointers
 * Fills buffer if _cached_bits is not enough to satisfy request
 */
void Bitstream::PeekNBits(bool *bits, const int N) {
	int i;
	int tempcurrent = _current;	// Temporary pointer to ensure _current is not modified

	if (!_eof & (N > _cached_bits)) FillBuffer();	// Fill the buffer if needed
	for (i = 0; i < N; i++) {
		if (_eof & (tempcurrent == _cache_end)) bits[i] = 0;	// Zero padding if EOF
		else {
			bits[i] = _buffer[tempcurrent++];
			if(tempcurrent == _buffer_size) tempcurrent = 0;	// Wrap around
		}
	}
}

/*
 * void Bitstream::ReadNBits(bool *bits, int N)
 *
 * Reads N Bits from input stream
 *
 * Requires :
 * *bits holds the output of this function (must be at least N large)
 * N is how many bits to read (must be less than _buffer_size)
 *
 * Modifies :
 * *bits will hold the next N bits.
 * Once EOF is reached, zeros will be padded into *bits
 * Moves _current to always point to the next bit to be read in
 * Moves _cached_bits to reflect reduction in the cache
 * 
 * Ensures :
 * Allows user to read a number of bits
 * Fills buffer if _cached_bits is not enough to satisfy request
 */
void Bitstream::ReadNBits(bool *bits, const int N) {
	int i;

	if (!_eof & (N > _cached_bits)) FillBuffer();	// Fill the buffer if needed
	for (i = 0; i < N; i++) {
		if (_eof & (_current == _cache_end)) bits[i] = 0;	// Zero padding if EOF
		else {
			bits[i] = _buffer[_current++];
			if(_current == _buffer_size) _current = 0;	// Wrap around
			_cached_bits--;
		}
	}
}

/*
 * void Bitstream::WriteNBits(bool *bits, int N)
 *
 * Writes N Bits into stream
 *
 * Requires :
 * *bits holds the bits to be written (must be at least N large)
 * N is how many bits are to be written (must be less than _buffer_size)
 * _outfile to be valid and open
 *
 * Modifies :
 * _buffer will hold the N bits.
 * Moves _current to always point to the next bit to be written into
 * Moves _cached_bits to reflect increase in the cache
 * 
 * Ensures :
 * Allows user to write a number of bits
 * Flushes buffer if buffer cannot hold anymore bits
 *
 * Warning:
 * Only can work for N < (_buffer_size - 7) for worst-case scenario
 */
void Bitstream::WriteNBits(const bool *bits, const int N) {
	int i;

	if (N > (_buffer_size - _cached_bits)) Flush(false);	// Flush the buffer if needed
	for (i = 0; i < N; i++) {
		_buffer[_current++] = bits[i];
		if (_current == _buffer_size) _current = 0;	// Wrap around
		_cached_bits++;
	}
}

/*
 * bool Bitstream::ReadBit()
 *
 * Reads one bit from stream
 *
 * Requires :
 * nothing
 *
 * Modifies :
 * nothing
 *
 * Ensures :
 * Allows user to read only one bit
 * Calls ReadNBits();
 */
bool Bitstream::ReadBit()
{
	bool b;
	ReadNBits(&b, 1);
	return b;
}

/*
 * void Bitstream::FillBuffer()
 *
 * Fills the Buffer from input file stream
 *
 * Requires :
 * Valid and open input file stream
 *
 * Modifies :
 * _buffer will hold reads_needed*8 bits
 * _cached_bits will increase by reads_needed*8 bits
 * _cache_end will increase by reads_needed*8 bits
 *
 * Ensures :
 * _buffer will be filled to capacity to reduce file accesses
 */
void Bitstream::FillBuffer() {
	unsigned char c;
	int reads_needed;
	int i,j;

	reads_needed = (_buffer_size - _cached_bits) >> 3; // number of char reads
	for (i = 0; i < reads_needed; i++) {	// Do the reading
		if (_cache_end == _buffer_size) _cache_end = 0;	// Wrap around
		_infile->get(c); // Grab character
		if (_infile->eof()) {	// If EOF is reached
			_eof = true;
			return;
		}
		else {
			for (j = 0; j < 8; j++) {
				_buffer[_cache_end++] = ((c & 128) > 0);	// bits read left to right
				c <<= 1;
			}
		}
		_cached_bits += 8;
	}
}

/*
 * void Bitstream::Flush(const bool all)
 *
 * Flushes the Buffer into output file stream
 *
 * Requires :
 * all of 1 will flush even the last unfinished byte (only for closing of file stream)
 *
 * Modifies :
 * The output file stream will read in reads_needed*8 bits
 * _cached_bits will decrease by writes_needed*8 bits
 * _cache_end will decrease by writes_needed*8 bits
 *
 * Ensures :
 * _buffer will be emptied if all is true
 * otherwise, _buffer will be emptied up to the last unfinished byte
 */
void Bitstream::Flush(const bool all) {
	unsigned char c = 0;
	int writes_needed;
	int i,j;
	HWND hWndProgress;

    hWndProgress = GetDlgItem(ghWndStatusDlg, IDC_TRANSFERPROGRESS);

	/********************/
	/* For Thread Begin */
	/********************/
	char * lpDataBuf;
	WRITEREQUEST* pWrite;

	/******************/
	/* For Thread End */
	/******************/

	if (all) {	// If flushing everything, better complete this byte
		j = 8 - (_current & 7);
		if (j != 8) for (i = 0; i < j; i++) {	// Check if current bit is byte-aligned
			_buffer[_current++] = 0;
			_cached_bits++;
		}
	}

	writes_needed = _cached_bits >> 3;	// number of char writes
	/*********************/
	/* From Thread Begin */
	/*********************/		
	// transfer file, loop until all blocks of file have been read
	lpDataBuf = (char *) HeapAlloc(hDataHeap, 0, writes_needed);
	pWrite = (WRITEREQUEST*) HeapAlloc(ghWriterHeap, 0, sizeof(WRITEREQUEST));

	while ((lpDataBuf == NULL) || (pWrite == NULL)) {
		BOOL fRes;
		
		//  Either the data heap is full, or the writer heap is full.
		//  Free any allocated block, wait a little and try again.
		
		//  Waiting lets the writer thread send some blocks and free
		//  the data blocks from the data heap and the control
		//  blocks from the writer heap.
		
		if (lpDataBuf) {
			EnterCriticalSection(&gcsDataHeap);
			fRes = HeapFree(hDataHeap, 0, lpDataBuf);
			LeaveCriticalSection(&gcsDataHeap);
			if (!fRes)
				ErrorReporter("HeapFree (Data block)");
		}
		
		if (pWrite) {
			EnterCriticalSection(&gcsWriterHeap);
			fRes = HeapFree(ghWriterHeap, 0, pWrite);
			LeaveCriticalSection(&gcsWriterHeap);
			if (!fRes)
				ErrorReporter("HeapFree (Writer block)");
		}
		
		OutputDebugString("Xfer: A heap is full.  Waiting...\n");
		
		// wait a little
		// check for abort during the wait
		if (WaitForSingleObject(hTransferAbortEvent, 200) == WAIT_OBJECT_0)
			Encoder_Abort = true;

		lpDataBuf = (char*) HeapAlloc(hDataHeap, 0, writes_needed);
		pWrite = (WRITEREQUEST*) HeapAlloc(ghWriterHeap, 0, sizeof(WRITEREQUEST));
	}
	/*******************/
	/* From Thread End */
	/*******************/

	for (i = 0; i < writes_needed; i++) {	// Do the writing
		if (_cache_end == _buffer_size) _cache_end = 0;	// Wrap around
		for (j = 0; j < 8; j++) {
			c = (unsigned char) ((c << 1) + _buffer[_cache_end++]);	// Bytes written left to right
		}

		_outfile->put(c);
                           /***************/
		lpDataBuf[i] = c;  /* From Thread */
                           /***************/
		_cached_bits -= 8;
	}

	/*********************/
	/* From Thread Begin */
	/*********************/
	WriterAddExistingNode(pWrite, WRITE_FILE, writes_needed, 0, lpDataBuf, hDataHeap, hWndProgress);
				
		//		if (dwRead != dwPacketSize) // eof
		//			break;
		
	// has the user aborted?
	if (WaitForSingleObject(hTransferAbortEvent, 0) == WAIT_OBJECT_0)
		Encoder_Abort = true;
	/*******************/
	/* From Thread End */
	/*******************/

}

#endif
