/*-----------------------------------------------------------------------------

    This is a part of the Microsoft Source Code Samples. 
    Copyright (C) 1995 Microsoft Corporation.
    All rights reserved. 
    This source code is only intended as a supplement to 
    Microsoft Development Tools and/or WinHelp documentation.
    See these sources for detailed information regarding the 
    Microsoft samples programs.

    MODULE: Reader.c

    PURPOSE: Read from comm port

    FUNCTIONS:
        OutputABufferToWindow - process incoming data destined for tty window
        OutputABufferToFile   - process incoming data destined for a file
        OutputABuffer         - called when data is read from port

-----------------------------------------------------------------------------*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include "Serialdecoder.h"

// Global variables from Serialdecoder.cpp
extern TTYInfoStruct TTYInfo;
extern HWND          ghWndStatusDlg;
extern DWORD  gdwReceiveState;
extern HANDLE ghFileCapture;

extern CRITICAL_SECTION gcsReaderHeap;
extern CRITICAL_SECTION gcsReadDataHeap;
extern CRITICAL_SECTION gcsHeapClear;
extern HANDLE ghReaderHeap;
extern HANDLE ghReadDataHeap;
extern WRITEREQUEST *gpReaderHead;
extern WRITEREQUEST *gpReaderTail;
extern BOOL gfAbortTransfer;

// to be removed
extern HWND ghWndTTY;

/*
    Prototypes for functions call only within this file
*/
void OutputABufferToHeap( HANDLE, char *, DWORD );


/*-----------------------------------------------------------------------------

FUNCTION: OutputABufferToWindow(HWND, char *, DWORD)

PURPOSE: Updates TTY Buffer with characters just received.

PARAMETERS:
    hTTY     - handle to the TTY child window
    lpBuf    - address of data buffer
    dwBufLen - size of data buffer

HISTORY:   Date       Author      Comment
            5/ 8/91   BryanW      Wrote it
           10/27/95   AllenD      Modified for MTTTY Sample

-----------------------------------------------------------------------------*/
void OutputABufferToWindow(HWND hTTY, char * lpBuf, DWORD dwBufLen)
{
    RECT rect;

    /*
        update screen buffer with new buffer
        need to do a character by character check
        for special characters
    */
    int i;

    for ( i = 0 ; i < (int) dwBufLen; i++) {
        switch (lpBuf[ i ]) {
            case ASCII_BEL:                // BELL CHAR
                MessageBeep( 0 ) ;
                break ;

            case ASCII_BS:                 // Backspace CHAR
                if (COLUMN( TTYInfo ) > 0)
                   COLUMN( TTYInfo ) -- ;
                break ;

            case ASCII_CR:                 // Carriage Return
                COLUMN( TTYInfo ) = 0 ;
                if (!NEWLINE( TTYInfo ))
                    break;

                //
                // FALL THROUGH
                //

            case ASCII_LF:                 // Line Feed
                if (ROW( TTYInfo )++ == MAXROWS - 1)
                {
                    MoveMemory( (LPSTR) (SCREEN( TTYInfo )),
                                  (LPSTR) (SCREEN( TTYInfo ) + MAXCOLS),
                                  (MAXROWS - 1) * MAXCOLS ) ;
                    FillMemory((LPSTR) (SCREEN( TTYInfo ) + (MAXROWS - 1) * MAXCOLS),
                                  MAXCOLS,  ' ' ) ;
                    InvalidateRect( hTTY, NULL, FALSE ) ;
                    ROW( TTYInfo )-- ;
                }
                break ;

            default:                       // standard character
                SCREENCHAR(TTYInfo, COLUMN(TTYInfo), ROW(TTYInfo)) = lpBuf[ i ];

                rect.left = (COLUMN( TTYInfo ) * XCHAR( TTYInfo )) -
                            XOFFSET( TTYInfo ) ;
                rect.right = rect.left + XCHAR( TTYInfo ) ;
                rect.top = (ROW( TTYInfo ) * YCHAR( TTYInfo )) -
                           YOFFSET( TTYInfo ) ;
                rect.bottom = rect.top + YCHAR( TTYInfo ) ;
                InvalidateRect( hTTY, &rect, FALSE ) ;

                // 
                // Line wrap
                //
                if (COLUMN( TTYInfo ) < MAXCOLS-1 )
                    COLUMN( TTYInfo )++ ;
                else if (AUTOWRAP( TTYInfo ))
                    OutputABufferToWindow(hTTY, "\r\n", 2 ) ;
                
                break;
        }
    }

    MoveTTYCursor(hTTY);
    return;
}

/*-----------------------------------------------------------------------------

FUNCTION: OutputABufferToFile(HANDLE, char *, DWORD)

PURPOSE: Output a rec'd buffer to a file

PARAMETERS:
    hFile    - handle of file save data into
    lpBuf    - address of data buffer
    dwBufLen - size of data buffer

HISTORY:   Date:      Author:     Comment:
           10/27/95   AllenD      Wrote it

-----------------------------------------------------------------------------*/
void OutputABufferToHeap(char *lpBuf, DWORD dwBufLen)
{
	WRITEREQUEST *pWrite;
	WRITEREQUEST *pOldLast;
	char *lpDataBuf;
	
	// transfer file, loop until all blocks of file have been read
	lpDataBuf = (char *) HeapAlloc(ghReadDataHeap, 0, dwBufLen);
	pWrite = (WRITEREQUEST *) HeapAlloc(ghReaderHeap, 0, sizeof(WRITEREQUEST));
	while ((lpDataBuf == NULL) || (pWrite == NULL)) {
		BOOL fRes;
		
		// Either the data heap is full, or the writer heap is full.
		// Free any allocated block, wait a little and try again.
		
		//  Waiting lets the writer thread send some blocks and free
		//  the data blocks from the data heap and the control
		//  blocks from the writer heap.
		
		if (lpDataBuf) {
			EnterCriticalSection(&gcsReadDataHeap);
			fRes = HeapFree(ghReadDataHeap, 0, lpDataBuf);
			LeaveCriticalSection(&gcsReadDataHeap);
			if (!fRes)
				ErrorReporter("HeapFree (Read Data block)");
		}
		
		if (pWrite) {
			EnterCriticalSection(&gcsReaderHeap);
			fRes = HeapFree(ghReaderHeap, 0, pWrite);
			LeaveCriticalSection(&gcsReaderHeap);
			if (!fRes)
				ErrorReporter("HeapFree (Reader block)");
		}
		
		OutputDebugString("Xfer: A heap is full.  Waiting...\n");
		
		// wait a little
		// check for abort during the wait
		Sleep(200);
		if (gfAbortTransfer) {
			OutputABufferToWindow(ghWndTTY, "Aborting Heap Allocator\n\r", strlen("Aborting Heap Allocator\n\r"));
			break;
		}
		lpDataBuf = (char*) HeapAlloc(ghReadDataHeap, 0, dwBufLen);
		pWrite = (WRITEREQUEST*) HeapAlloc(ghReaderHeap, 0, sizeof(WRITEREQUEST));
	}
	if (!gfAbortTransfer) {

		char *ptr = lpDataBuf;
		char *ptr2 = lpBuf;
		for (DWORD i = 0; i < dwBufLen; i++)
			*ptr++ = *ptr2++;

/*		ptr = lpDataBuf;
		FILE *fileptr;
		fileptr = fopen("c:/winnt/profiles/jchuang/desktop/compiler/output.bits", "a+b");
		for (i = 0; i < dwBufLen; i++)
			fprintf(fileptr, "%c", *ptr++);
		fclose(fileptr);
		
		HeapFree(ghReadDataHeap, 0, lpDataBuf);
*/
		// WriterAddExistingNode(pWrite, WRITE_FILE, dwRead, 0, lpDataBuf, ghReadDataHeap, NULL);
		pWrite->dwWriteType  = WRITE_FILE;
		pWrite->dwSize       = dwBufLen;
		pWrite->ch           = 0;
		pWrite->lpBuf        = lpDataBuf;
		pWrite->hHeap        = ghReadDataHeap;
		pWrite->hWndProgress = NULL;

			//			wsprintf(output, "pWrite->lpBuf: %s.\n\r", pWrite->lpBuf);
//	OutputABufferToWindow( ghWndTTY, output, strlen(output) );
//			wsprintf(output, "lpDataBuf: %s.\n\r", lpDataBuf);
//	OutputABufferToWindow( ghWndTTY, output, strlen(output) );

		//
		// Add to Linked list
		//
		EnterCriticalSection(&gcsReaderHeap);
		pOldLast = gpReaderTail->pPrev;
		
		pWrite->pNext = gpReaderTail;
		pWrite->pPrev = pOldLast;
		
		pOldLast->pNext = pWrite;
		gpReaderTail->pPrev = pWrite;
		LeaveCriticalSection(&gcsReaderHeap);
		
		// has the user aborted?
		//if (WaitForSingleObject(hTransferAbortEvent, 0) == WAIT_OBJECT_0)
		
		//
		// update transfer progress bar
		//
		PostMessage(GetDlgItem(ghWndStatusDlg, IDC_TRANSFERPROGRESS), PBM_STEPIT, 0, 0);
	}

	return;
}

/*-----------------------------------------------------------------------------

FUNCTION: OutputABuffer(HWND, char *, DWORD)

PURPOSE: Send a rec'd buffer to the approprate location

PARAMETERS:
    hTTY     - handle to the TTY child window
    lpBuf    - address of data buffer
    dwBufLen - size of data buffer

COMMENTS: If buffer is 0 length, then do nothing.

HISTORY:   Date:      Author:     Comment:
           10/27/95   AllenD      Wrote it

-----------------------------------------------------------------------------*/
void OutputABuffer(HWND hTTY, char * lpBuf, DWORD dwBufLen)
{
    if (dwBufLen == 0) {
        OutputDebugString("NULL Buffer in OutputABuffer\n\r");
        return;
    }

    switch(gdwReceiveState)
    {
        case RECEIVE_TTY:
            OutputABufferToWindow(hTTY, lpBuf, dwBufLen);
            break;

        case RECEIVE_CAPTURED:
			if (!gfAbortTransfer) {
				EnterCriticalSection(&gcsHeapClear);
				if ((ghReaderHeap != NULL) && (ghReadDataHeap != NULL)) {
					OutputABufferToHeap(lpBuf, dwBufLen);
				}
				LeaveCriticalSection(&gcsHeapClear);
			}

            break;

        default:
            OutputDebugString("Unknown receive state!\n\r");
    }

    return;
}
